<?php

namespace app\models;

use Yii;
use yii\base\Model;

class RegisterForm extends Model
{
    public $full_name;
    public $phone;
    public $email;
    public $address;
    public $residents_count;
    public $password;
    public $password_repeat;
    public $agree_terms;

    public function rules()
    {
        return [
            // Обязательные поля
            [['full_name', 'phone', 'email', 'address', 'residents_count', 'password', 'password_repeat'], 'required', 'message' => 'Это поле обязательно для заполнения'],
            ['agree_terms', 'required', 'requiredValue' => 1, 'message' => 'Вы должны согласиться с правилами регистрации'],
            
            // Валидация форматов
            ['full_name', 'match', 'pattern' => '/^[а-яА-ЯёЁ\s]+$/u', 'message' => 'ФИО должно содержать только кириллические символы и пробелы'],
            ['phone', 'match', 'pattern' => '/^8\(\d{3}\)\d{3}-\d{2}-\d{2}$/', 'message' => 'Формат телефона: 8(XXX)XXX-XX-XX'],
            ['email', 'email', 'message' => 'Некорректный формат email'],
            ['residents_count', 'integer', 'min' => 1, 'message' => 'Количество проживающих должно быть целым числом'],
            
            // Валидация пароля
            ['password', 'string', 'min' => 10, 'message' => 'Пароль должен содержать минимум 10 символов'],
            ['password_repeat', 'compare', 'compareAttribute' => 'password', 'message' => 'Пароли не совпадают'],
            
            // Уникальность
            ['phone', 'validateUniquePhone'],
            ['email', 'validateUniqueEmail'],
        ];
    }

    public function validateUniquePhone($attribute, $params)
    {
        if (User::find()->where(['phone' => $this->phone])->exists()) {
            $this->addError($attribute, 'Этот телефон уже зарегистрирован');
        }
    }

    public function validateUniqueEmail($attribute, $params)
    {
        if (User::find()->where(['email' => $this->email])->exists()) {
            $this->addError($attribute, 'Этот email уже зарегистрирован');
        }
    }

    public function register()
    {
        if ($this->validate()) {
            $user = new User();
            $user->scenario = User::SCENARIO_REGISTER;
            
            // Копируем атрибуты
            $user->full_name = $this->full_name;
            $user->phone = $this->phone;
            $user->email = $this->email;
            $user->address = $this->address;
            $user->residents_count = $this->residents_count;
            $user->password = $this->password;
            $user->agree_terms = $this->agree_terms;
            
            if ($user->save()) {
                return $user;
            } else {
                // Добавляем ошибки из модели User в форму
                foreach ($user->getErrors() as $attribute => $errors) {
                    foreach ($errors as $error) {
                        $this->addError($attribute, $error);
                    }
                }
            }
        }
        return false;
    }

    public function attributeLabels()
    {
        return [
            'full_name' => 'ФИО',
            'phone' => 'Телефон',
            'email' => 'Email',
            'address' => 'Адрес проживания',
            'residents_count' => 'Количество проживающих',
            'password' => 'Пароль',
            'password_repeat' => 'Повтор пароля',
            'agree_terms' => 'Согласие с правилами регистрации',
        ];
    }
}